<?php defined('BASEPATH') OR exit('No direct script access allowed');

// This can be removed if you use __autoload() in config.php OR use Modular Extensions
require APPPATH.'/libraries/DaaS_REST_Controller.php';

class Direct extends DaaS_REST_Controller
{	
	/* SEND (POST request only)
	 * This function sends a direct message utilizing the passed POST parameters if the
	 * authentication is successfully and all required parameters have been passed.
	 */
	public function send_post() {
		$this->load->library(array('email','permissions'));
		$this->load->helper('mail');
		$this->load->model('requestmodel');
		$request_id = $this->create_request($this->request->uri . ' ' . strtoupper($this->request->method));
		$response_message['request_id'] = $request_id;
		
		$error = false;
		$fields = array();
		
		$is_authenticated = $this->request->hmac_authenticated; //check if authenticated request
		$is_authorized = $this->request->direct_api_authorized; //authorize request
		if (!$is_authenticated) {
			$this->email->clear(true);
			$response_message['message'] = 'Access Denied. Authentication Failed.';
			$this->requestmodel->add_response($request_id, 401, json_encode($response_message));
			$this->response($response_message, 401);
		}
		
		if (!$is_authorized) {
			$this->email->clear(true);
			$response_message['message'] = 'Access Denied. Use Not Authorized.';
			$this->requestmodel->add_response($request_id, 403, json_encode($response_message));
			$this->response($response_message, 403);
		}
		
		//check parameters for required fields, set defaults for non-required fields that aren't provided
		if(!$this->post('sender')) { $error = true; array_push($fields, 'sender'); } else { $sender = $this->post('sender'); }
		if(!$this->post('to')) { $error = true; array_push($fields, 'to'); } else { $to = $this->post('to'); }
		if(!$this->post('cc')) { $cc = null; } else { $cc = $this->post('cc'); }
		if(!$this->post('mailtype')) { $mailtype = 'html'; } else { $mailtype = $this->post('mailtype'); }
		if(!$this->post('priority')) { $priority = 3; } else { $priority = $this->post('priority'); }
		if(!$this->post('subject')) { $subject = '(No Subject)'; } else { $subject = $this->post('subject'); }
		if(!$this->post('body')) { $body = ''; } else { $body = $this->post('body'); }
		
		//if there is an error with the passed in fields create an error message
		if ($error){
			$response_message['message'] = $this->generate_required_fields_message($fields);
			$response_message['fields'] = $fields;
			$this->requestmodel->add_response($request_id, 422, json_encode($response_message));
			$this->response($response_message, 422);
		}
		
		//check that this app is authorized to send a message on behalf of this user
		$this->load->model('applicationmodel');
		$query = $this->applicationmodel->get_application_from_public($this->request->public_key);
		$query_array = $query->row_array();
		$app_id = $query_array['id'];
		$permissions = $this->permissions->get_user_permissions_for_address($sender,$app_id);
		if(!isset($permissions['send']) || $permissions['send'] !== TRUE) {
			$this->email->clear(true);
			$response_message['message'] = 'Access Denied. User Has Not Authorized.';
			$this->requestmodel->add_response($request_id, 403, json_encode($response_message));
			$this->response($response_message, 403);
		}
		
		//parse multiple addresses in the to string
		$address_arr = array();
		$to_arr = array();//array for storing to for the send
		$cc_arr = array();//array for storing cc for the send
		
		//this call adds addresses to the address array for validation and to array and cc array for actual mail
		normalize_address($to, $to_arr, $address_arr);
		if(isset($cc)) { normalize_address($cc, $cc_arr, $address_arr); }
		
		//mail configuration items
        $mailconfig['smtp_host'] = GATEWAY_SMTP_HOSTNAME;
        $mailconfig['smtp_port'] = GATEWAY_SMTP_PORT;
        $mailconfig['mailtype'] = $mailtype;
        $mailconfig['protocol'] = DIRECT_SEND_PROTOCOL;
        $mailconfig['smtp_timeout'] = GATEWAY_SMTP_TIMEOUT;
        $mailconfig['priority'] = $priority;

		$this->email->initialize($mailconfig);
        $this->email->set_newline("\r\n");
		
		$this->email->from($sender); //set From line
		
		$to_arr = array_filter($to_arr, 'strlen');
		$this->email->to($to_arr); //set To line
		
		if(isset($cc)) {
			$cc_arr = array_filter($cc_arr, 'strlen');
			$this->email->cc($cc_arr); //set CC line
		}
		
        $this->email->subject($subject); //set Subject line
        $this->email->message($body); //set Body of message
	
		//get attachments
		$i = 0;
		$totalsize = 0;
		foreach (array_keys($_FILES) as $value){
		   if (isset($_FILES[$value])){
				if(isset($_FILES[$value]['binary'])) { $binary_str = $_FILES[$value]['binary']; } //if $_FILES was set manually, grab binary directly
				else { $binary_str = file_get_contents($_FILES[$value]['tmp_name']); } //otherwise get binary from file
				$extension = pathinfo($_FILES[$value]['name'], PATHINFO_EXTENSION);
				$filesize = $_FILES[$value]['size'];
				$totalsize = $totalsize + $filesize;
				if (0 == strcasecmp($extension, 'exe')){
					$response_message['message'] = $_FILES[$value]['name'].' is an invalid file type.';
					$this->requestmodel->add_response($request_id, 422, json_encode($response_message));
					$this->response($response_message, 422);
				}
				$this->email->string_attach($binary_str, $_FILES[$value]['name']); //set Attachment
		   }
		}

		//if the size is over the limit, prepare an error response
		if ($totalsize > DIRECT_ATTACHMENT_SIZE_LIMIT){
			$this->load->helper('number');
			$response_message['message'] = 'The total file size is greater than '.byte_format(DIRECT_ATTACHMENT_SIZE_LIMIT,0).'.';
			$this->requestmodel->add_response($request_id, 413, json_encode($response_message));
			$this->response($response_message, 413);
		}
		
		//check if any of the addresses are invalid and add them to this array
		$invalid_addresses = array();
		$valid_message = true;
		foreach($address_arr as $address){
			$is_valid = $this->validate($sender,$address);
			//if there is no cert for that address add it to the invalid array
			if (!$is_valid){
				array_push($invalid_addresses, $address);
				$valid_message = false;
			}
		}

		//loop through the invalid addresses array and create a formatted message
		if (!$valid_message){
			if($is_authenticated && $is_authorized) { 
				$response_message['message'] = $this->generate_invalid_recipients_message($invalid_addresses);
				$this->requestmodel->add_response($request_id, 400, json_encode($response_message));
				$this->response($response_message, 400); 
			}
		}
		
		//if authentication was successful, attempt to send message
		if(!$this->email->send()) {
			$this->email->clear(true);
			$response_message['message'] = 'Message failed to send.';
			$this->requestmodel->add_response($request_id, 400, json_encode($response_message));
			$this->response($response_message, 400);
		}
		else { 
			$this->email->clear(true);
			$response_message['message'] = 'Message sent.';
			$this->requestmodel->add_response($request_id, 200, json_encode($response_message));
			$this->response($response_message, 200); 
		}
	}

	/* VALIDATE (GET request only)
	 * This function validates whether the provided address or domain is trusted by the
	 * system.
	 */
	public function validate_get() {
		$this->load->model('requestmodel');
		$this->load->helper('mail');
		$request_id = $this->create_request($this->request->uri . ' ' . strtoupper($this->request->method));
		$response_message['request_id'] = $request_id;
		
		$is_authenticated = $this->request->hmac_authenticated; //check if authenticated request
		$is_authorized = $this->request->direct_api_authorized; //authorize request
		
		if(!$is_authenticated) {
			$response_message['message'] = 'Access Denied. Authentication Failed.';
			$this->requestmodel->add_response($request_id, 401, json_encode($response_message));
			$this->response($response_message, 401);
		}
		if(!$is_authorized) {
			$response_message['message'] = 'Access Denied. Use Not Authorized.';
			$this->requestmodel->add_response($request_id, 403, json_encode($response_message));
			$this->response($response_message, 403);
		}
		
		//if no address provided, return message that that field is required
		if(!$this->get('address')) { 
			$response_message['valid'] = false;
			$response_message['message'] = 'Must provide address field.';
			$this->requestmodel->add_response($request_id, 422, json_encode($response_message));
			$this->response($response_message, 422); 
		}
		else { $address = base64_decode(rawurldecode($this->get('address'))); }
		//if no sender provided, assume that there is an organizational cert that any domain address will work for
		if(!$this->get('sender')) { 
			$sender = 'validitytest@'.DIRECT_DOMAIN;
		}
		else { $sender = base64_decode(rawurldecode($this->get('sender'))); }
		
		//array for storing addresses to validate / storing multiple "To" addresses for the send function
		$address_arr = array();
		$to_arr = array(); 
		
		//this call adds addresses to the address array for validation and to array for actual mail
		normalize_address($address, $to_arr, $address_arr);
		
		$invalid_addresses = array();
		$valid_message = true;
			
		//call direct reference implementation web service to validate trust
		$url = 'http://'.GATEWAY_HOSTNAME.':8081/config-service/ConfigurationService?wsdl';
		try {
			$client = new SoapClient($url);
			foreach($address_arr as $address){
				$result = $client->validateTrust(array('sender'=>$sender,'address'=>$address));
				if(!$result->return) {
					array_push($invalid_addresses, $address);
					$valid_message = $result->return;
				}
			}
		}
		catch (Exception $e) {
			//If java web service fails, native PHP check if any of the addresses are invalid and add them to this array
			foreach($address_arr as $address){
				$has_cert = $this->has_trusted_certificate($address);
				//if there is no cert for that address add it to the invalid array
				if (!$has_cert){
					array_push($invalid_addresses, $address);
					$valid_message = $has_cert;
				}
			}
		}
		
		if($valid_message) { //if everything is valid, then return response	
			$response_message['valid'] = true;
			$this->requestmodel->add_response($request_id, 200, json_encode($response_message));			
			$this->response($response_message, 200); 
		}
		else { //if something is invalid, then return response
			$response_message['valid'] = false;
			$response_message['message'] = $this->generate_invalid_recipients_message($invalid_addresses);
			$this->requestmodel->add_response($request_id, 200, json_encode($response_message));
			$this->response($response_message, 200); 
		}
	}
	
	/* -----------------------------*
	 *  PRIVATE FUNCTIONS           *
	 * -----------------------------*/
	
	private function validate($sender,$address) {
		$address_arr = array();
		$to_arr = array();
		$this->load->helper('mail');
		//this call adds addresses to the address array for validation and to array for actual mail
		normalize_address($address, $to_arr, $address_arr);
		
		$invalid_addresses = array();
		$valid_message = true;
			
		//call direct reference implementation web service to validate trust
		$url = 'http://'.GATEWAY_HOSTNAME.':8081/config-service/ConfigurationService?wsdl';
		try {
			$client = new SoapClient($url);
			foreach($address_arr as $address){
				$result = $client->validateTrust(array('sender'=>$sender,'address'=>$address));
				if(!$result->return) {
					array_push($invalid_addresses, $address);	
					$valid_message = $result->return;
				}
			}
		}
		catch (Exception $e) {
			//If java web service fails, native PHP check if any of the addresses are invalid and add them to this array
			foreach($address_arr as $address){
				$has_cert = $this->has_trusted_certificate($address);
				//if there is no cert for that address add it to the invalid array
				if (!$has_cert){
					array_push($invalid_addresses, $address);
					$valid_message = $has_cert;
				}
			}
		}
		return $valid_message;
	}
	
	/* This function utilizes the config-service of the Direct Project Reference Implementation to
	 * check the trust chain of provided address. It can check if the address is trusted for outgoing
	 * or incoming messages individually by use of the directon parameter, but checks that it is trusted
	 * for both by default. It will return TRUE for a trusted address or domain and FALSE for an untrusted
	 * address or if trust cannot be checked due to a failure. IT calls validate_trust_chain for the heavy
	 * lifting on checking the trust chain.
	 */
	private function has_trusted_certificate($address,$direction = 'both'){
		//load helpers/libraries needed, connect to SOAP client
		$url = 'http://'.GATEWAY_HOSTNAME.':8081/config-service/ConfigurationService?wsdl';
		$client = new SoapClient($url);
		$this->load->helper('netdns2');
		$this->load->helper('mail');
		$this->load->helper('cert');
		
		//parse address so that we get just the address part
		$address = normalize_address_string($address);
		
		//get domain of address
		$domain = explode("@", $address);
		$domain = $domain[(count($domain)-1)];
		
		$cert = get_dns_cert($address);
		if(!$cert) { $cert = get_dns_cert($domain); } //if no cert for address, check for organizational cert
		
		//if still no cert, look for cert through web service TO-DO: we are repeating logic here, need to abstract this to a function or something
		if(!$cert) {
			$result = $client->getCertificatesForOwner(array('owner'=>$address));
			if(isset($result->return) && $result->return->status === 'ENABLED') {
				$parsed_cert = openssl_x509_parse($result->return->data);
				if(empty($parsed_cert)) { $pem_cert = convert_cert('DER','PEM',$result->return->data); } //if that didn't work, try converting it to PEM
				if(isset($pem_cert)) { $parsed_cert = openssl_x509_parse($pem_cert); } //parse converted cert
				$cert = $parsed_cert;
			}
		}
		if(empty($cert) || !$cert) { //if still no cert, look for organizational cert in web service
			$result = $client->getCertificatesForOwner(array('owner'=>$domain));
			if(isset($result->return) && $result->return->status === 'ENABLED') {
				$parsed_cert = openssl_x509_parse($result->return->data);
				if(empty($parsed_cert)) { $pem_cert = convert_cert('DER','PEM',$result->return->data); } //if that didn't work, try converting it to PEM
				if(isset($pem_cert)) { $parsed_cert = openssl_x509_parse($pem_cert); } //parse converted cert
				$cert = $parsed_cert;
			}
		} 
		//if we still can't find a certificate then they are not trusted
		if(empty($cert) || !$cert) { return FALSE; }
		
		//look for anchors in web service here
		$result = $client->listAnchors();
		$anchors = $result->return;
		$trusted = $this->validate_trust_chain($cert,$anchors,$direction);
		return $trusted;
	}
	

	/* This function recursively checks the trust chain of the provided certificate (cert parameter is
	 * expected to be an array result from openssl_x509_parse of a certificate). Anchors is data from the
	 * Direct Project RI config-service from the listAnchors function. 
	 */
	private function validate_trust_chain($cert,$anchors,$direction = 'both') {
		if((time() < $cert['validFrom_time_t']) || (time() > $cert['validTo_time_t'])) { return FALSE; } //check for certificate expiration/not yet valid-ness
		foreach($anchors as $anchor) {
			unset($parsed_anchor); unset($pem_anchor);
			$parsed_anchor = openssl_x509_parse($anchor->data);
			if(empty($parsed_anchor)) { $pem_anchor = convert_cert('DER','PEM',$anchor->data); } //if that didn't work, try converting it to PEM
			if(isset($pem_anchor)) { $parsed_anchor = openssl_x509_parse($pem_anchor); } //parse converted cert
			$anchor_cert = $parsed_anchor;
			
			//TO-DO: Check if certificates are revoked
			//check if cert issuer matches the anchor cert we are checking, if the current cert is not also the current anchor
			if($cert['issuer'] == $anchor_cert['subject'] && $cert['subject'] != $anchor_cert['subject']) {
				return $this->validate_trust_chain($anchor_cert,$anchors);
			}
			//if the current cert is issued by the current anchor, and the current cert is also the current anchor, we are at the root certificate
			else if($cert['issuer'] == $anchor_cert['subject'] && $cert['subject'] == $anchor_cert['subject']) { 
				if($direction === 'both' && $anchor->outgoing && $anchor->incoming && $anchor->status === 'ENABLED') { return TRUE; }
				else if($direction === 'incoming' && $anchor->incoming && $anchor->status === 'ENABLED') { return TRUE; }
				else if($direction === 'outgoing' && $anchor->outgoing && $anchor->status === 'ENABLED') { return TRUE; }
			}
		}
		return FALSE;
	}
}